using System;
using System.Data;
using System.Collections;

using gov.va.med.vbecs.DAL.VistALink.Client;
using gov.va.med.vbecs.DAL.VistALink.OpenLibrary;

namespace gov.va.med.vbecs.DAL.VAL
{	
	#region Header

	///<Package>Package: VBECS - VistA Blood Establishment Computer System</Package>
	///<Warning> WARNING: Per VHA Directive $VADIRECTIVE this class should not be modified</Warning>
	///<MedicalDevice> Medical Device #: $MEDDEVICENO</MedicalDevice>
	///<Developers>
	///	<Developer>Stas Antropov</Developer>
	///</Developers>
	///<SiteName>Hines OIFO</SiteName>
	///<CreationDate>2/27/2005</CreationDate>
	///<Note>The Food and Drug Administration classifies this software as a medical device.  As such, it may not be changed in any way. Modifications to this software may result in an adulterated medical device under 21CFR820, the use of which is considered to be a violation of US Federal Statutes.  Acquiring and implementing this software through the Freedom of information Act requires the implementor to assume total responsibility for the software, and become a registered manufacturer of a medical device, subject to FDA regulations</Note>
	///<summary>Provides means for getting provider information from VistA system via VistALink.</summary>

	#endregion

	public class Provider
	{
		private const string PROVIDER_LOOKUP_RPC_NAME = "VBECS PROVIDER LOOKUP";
		private const string PROVIDER_TABLE_NAME = "Provider";

		private Provider() {}

		///<Developers>
		///	<Developer>Brian Tomlin</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/17/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3929"> 
		///		<ExpectedInput>'589A5' as divisionCode, 'T' as providerName</ExpectedInput>
		///		<ExpectedOutput>Non-empty DataTable with provider data retrieved from VistA
		///		where the ProviderName starts with 'T'</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="3930"> 
		///		<ExpectedInput>'589' as divisionCode, Empty String for providerName</ExpectedInput>
		///		<ExpectedOutput>Non-empty DataTable with all Providers defined in VistA.</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="3931"> 
		///		<ExpectedInput>Null divisionCode, 'A' as providerName</ExpectedInput>
		///		<ExpectedOutput>ArgumentNullException</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="3932"> 
		///		<ExpectedInput>'589' as divisionCode, Null providerName</ExpectedInput>
		///		<ExpectedOutput>ArgumentNullException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Queries VistA system executing the VBECS PROVIDER LOOKUP RPC and returns
		/// a read-only <see cref="DataSet"/> containing search results. Search is
		/// performed by DivisionCode and optional Provider name parameters.
		/// </summary>
		/// <param name="divisionCode">Division code to search by.</param>
		/// <param name="providerName">Provider name to search by.</param>
		/// <returns>Read-Only <see cref="DataTable"/> containing provider lookup results.</returns>
		public static DataTable GetVistaProvider( string divisionCode, string providerName )
		{
			if( divisionCode == null )
				throw( new ArgumentNullException( "divisionCode" ) );

			if( providerName == null )
				throw( new ArgumentNullException( "providerName" ) );

			DataSet _ds = new DataSet();

			_ds = GetVistaProviderData( divisionCode, providerName );

			return _ds.Tables.Contains( PROVIDER_TABLE_NAME ) ? _ds.Tables[ PROVIDER_TABLE_NAME ] : _ds.Tables[0];
		}

		/// <summary>
		/// Executes Vista provider lookup RPC with give parameters and gets
		/// results in a read-only <see cref="DataSet"/>
		/// </summary>
		/// <param name="divisionCode">Division code string</param>
		/// <param name="searchValue">Provider Name string</param>
		/// <returns>Read-only <see cref="DataSet"/> containing provider lookup results</returns>
		private static DataSet GetVistaProviderData( string divisionCode, string searchValue )
		{
			string _rpcResult = VistALink.GetRpcBroker().ExecuteRpc( GetProviderLookupRpcRequest( divisionCode, searchValue ) );
			
			return DataTransformUtility.LoadXmlStringIntoDataSet( _rpcResult );
		}

		/// <summary>
		/// Creates provider lookup RPC request from the supplied parameters. 
		/// </summary>
		/// <param name="divisionCode">DivisionCode string</param>
		/// <param name="searchValue">Provider Name string</param>
		/// <returns>Provider Lookup RpcRequest</returns>
		private static RpcRequest GetProviderLookupRpcRequest( string divisionCode, string searchValue )
		{
			RpcRequest _providerLookupRequest = VistALink.CreateRpcRequest( PROVIDER_LOOKUP_RPC_NAME );

			_providerLookupRequest.Parameters.Add( new RpcStringParameter( 1, divisionCode ) );
			_providerLookupRequest.Parameters.Add( new RpcStringParameter( 2, ( searchValue != null ) ? searchValue : String.Empty ) );

			return _providerLookupRequest;
		}
	}
}
